@extends('layouts.app')

@php
use App\Models\Setting;
@endphp

@section('content')
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5>المعاملات المالية</h5>
        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addTransactionModal">
            إضافة معاملة جديدة
        </button>
    </div>
    <div class="card-body">
        <table class="table table-hover">
            <thead>
                <tr>
                    <th>#</th>
                    <th>التاريخ</th>
                    <th>النوع</th>
                    <th>التصنيف</th>
                    <th>المبلغ</th>
                    <th>الشريك</th>
                    <th>الوصف</th>
                    <th>الإجراءات</th>
                </tr>
            </thead>
            <tbody>
                @foreach($transactions as $transaction)
                <tr>
                    <td>{{ $transaction->id }}</td>
                    <td>{{ $transaction->transaction_date->format('Y-m-d') }}</td>
                    <td>
                        <span class="badge bg-{{ $transaction->type === 'income' ? 'success' : 'danger' }}">
                            {{ $transaction->type === 'income' ? 'إيراد' : 'مصروف' }}
                        </span>
                    </td>
                    <td>
                        <span class="badge" style="background-color: {{ $transaction->category_info['color'] }}">
                            <i class="bi {{ $transaction->category_info['icon'] }}"></i>
                            {{ $transaction->category_info['name'] }}
                        </span>
                    </td>
                    <td>{{ number_format($transaction->amount, 3) }} {{ Setting::get('currency', 'JOD') }}</td>
                    <td>{{ $transaction->partner->name ?? 'معاملة عامة' }}</td>
                    <td>{{ $transaction->description }}</td>
                    <td>
                        <div class="btn-group">
                            @can('transactions-edit')
                            <button type="button" class="btn btn-sm btn-primary edit-transaction" 
                                    data-id="{{ $transaction->id }}"
                                    data-type="{{ $transaction->type }}"
                                    data-category="{{ $transaction->category }}"
                                    data-amount="{{ $transaction->amount }}"
                                    data-description="{{ $transaction->description }}"
                                    data-date="{{ $transaction->transaction_date->format('Y-m-d') }}"
                                    data-partner="{{ $transaction->partner_id }}">
                                <i class="bi bi-pencil"></i>
                            </button>
                            @endcan
                            @can('transactions-delete')
                            <button type="button" class="btn btn-sm btn-danger delete-transaction" data-id="{{ $transaction->id }}">
                                <i class="bi bi-trash"></i>
                            </button>
                            @endcan
                        </div>
                    </td>
                </tr>
                @endforeach
            </tbody>
        </table>
    </div>
</div>

<!-- Modal إضافة معاملة -->
<div class="modal fade" id="addTransactionModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">إضافة معاملة جديدة</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="addTransactionForm">
                @csrf
                <div class="modal-body">
                    @role(['admin', 'accountant'])
                    <div class="mb-3">
                        <label class="form-label">الشريك</label>
                        <select class="form-select" name="partner_id">
                            <option value="">معاملة عامة</option>
                            @foreach($partners as $partner)
                                <option value="{{ $partner->id }}">{{ $partner->name }}</option>
                            @endforeach
                        </select>
                    </div>
                    @endrole

                    <div class="mb-3">
                        <label class="form-label">النوع <span class="text-danger">*</span></label>
                        <select class="form-select" name="type" id="transaction_type" required>
                            <option value="">اختر النوع</option>
                            <option value="income">إيراد</option>
                            <option value="expense">مصروف</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">التصنيف <span class="text-danger">*</span></label>
                        <select class="form-select" name="category" id="transaction_category" required>
                            <option value="">اختر التصنيف</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">المبلغ</label>
                        <input type="number" class="form-control" name="amount" min="0" step="0.01" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">التاريخ</label>
                        <input type="date" class="form-control" name="transaction_date" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">الوصف</label>
                        <textarea class="form-control" name="description"></textarea>
                    </div>
                    <div class="buyer-info" style="display: none;">
                        <hr>
                        <h6>معلومات المشتري</h6>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">اسم المشتري <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" name="buyer_name">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">رقم الهاتف</label>
                                <input type="text" class="form-control" name="buyer_phone">
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">العنوان</label>
                                <input type="text" class="form-control" name="buyer_address">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">الرقم الضريبي</label>
                                <input type="text" class="form-control" name="buyer_tax_number">
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">إلغاء</button>
                    <button type="submit" class="btn btn-primary">حفظ</button>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
$(document).ready(function() {
    // إضافة CSRF token لجميع طلبات AJAX
    $.ajaxSetup({
        headers: {
            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
        }
    });

    // تحديث التصنيفات عند تغيير النوع
    $('#transaction_type').on('change', function() {
        const type = $(this).val();
        const categorySelect = $('#transaction_category');
        categorySelect.empty().append('<option value="">اختر التصنيف</option>');
        
        if (type) {
            const categories = @json($categories);
            const typeCategories = categories[type] || {};
            
            Object.entries(typeCategories).forEach(([value, category]) => {
                categorySelect.append(`
                    <option value="${value}" data-icon="${category.icon}" data-color="${category.color}">
                        ${category.name}
                    </option>
                `);
            });
        }
    });

    // تحديث نموذج إضافة معاملة
    $('#addTransactionForm').on('submit', function(e) {
        e.preventDefault();
        $.ajax({
            url: '{{ route("transactions.store") }}',
            method: 'POST',
            data: $(this).serialize(),
            success: function(response) {
                $('#addTransactionModal').modal('hide');
                showAlert('success', response.message);
                setTimeout(() => location.reload(), 1500);
            },
            error: function(xhr) {
                showAlert('error', xhr.responseJSON?.message || 'حدث خطأ أثناء إضافة المعاملة');
            }
        });
    });

    // تحديث نموذج تعديل معاملة
    $('#editTransactionForm').on('submit', function(e) {
        e.preventDefault();
        const id = $('#edit_transaction_id').val();
        $.ajax({
            url: `/transactions/${id}`,
            method: 'PUT',
            data: $(this).serialize(),
            success: function(response) {
                $('#editTransactionModal').modal('hide');
                showAlert('success', response.message);
                setTimeout(() => location.reload(), 1500);
            },
            error: function(xhr) {
                showAlert('error', xhr.responseJSON?.message || 'حدث خطأ أثناء تحديث المعاملة');
            }
        });
    });

    // حذف معاملة
    window.deleteTransaction = function(id) {
        if (confirm('هل أنت متأكد من حذف هذه المعاملة؟')) {
            $.ajax({
                url: `/transactions/${id}`,
                method: 'DELETE',
                success: function(response) {
                    showAlert('success', response.message);
                    setTimeout(() => location.reload(), 1500);
                },
                error: function(xhr) {
                    showAlert('error', xhr.responseJSON?.message || 'حدث خطأ أثناء حذف المعاملة');
                }
            });
        }
    };

    // دالة مساعدة لعرض التنبيهات
    function showAlert(type, message) {
        const alertDiv = $('<div>')
            .addClass(`alert alert-${type} alert-dismissible fade show`)
            .attr('role', 'alert')
            .html(`
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `);

        $('.container').prepend(alertDiv);
        
        setTimeout(() => {
            alertDiv.alert('close');
        }, 3000);
    }

    // إظهار/إخفاء معلومات المشتري
    function toggleBuyerInfo() {
        const type = $('#transaction_type').val();
        const shouldShow = type === 'income';
        
        $('.buyer-info').toggle(shouldShow);
        $('.buyer-info input[name="buyer_name"]').prop('required', shouldShow);
    }

    $('#transaction_type, #transaction_category').on('change', toggleBuyerInfo);
});
</script>
@endpush 