<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use App\Models\Partner;
use App\Models\Setting;
use Illuminate\Http\Request;

class ReportsController extends Controller
{
    public function index()
    {
        // المحاسب والمدير يريان جميع الشركاء، الشريك يرى نفسه فقط
        $partners = auth()->user()->hasRole(['admin', 'accountant']) 
            ? Partner::all()
            : Partner::where('id', auth()->user()->partner_id)->get();

        $query = Transaction::query();
        
        // إذا كان المستخدم شريك، نظهر فقط معاملاته والمعاملات العامة
        if (!auth()->user()->hasRole(['admin', 'accountant'])) {
            $query->where(function($q) {
                $q->where('partner_id', auth()->user()->partner_id)
                  ->orWhereNull('partner_id');
            });
        }

        // حساب الإجماليات
        $totalIncome = $query->clone()->where('type', 'income')->sum('amount');
        $totalExpense = $query->clone()->where('type', 'expense')->sum('amount');
        $netProfit = $totalIncome - $totalExpense;

        // حساب حصص الشركاء
        $isPartner = !auth()->user()->hasRole(['admin', 'accountant']);
        $allPartners = Partner::all();
        $totalShares = $allPartners->sum('share');
        
        $partnerShares = [];
        
        // إذا كان المستخدم مدير أو محاسب، نضيف جميع الشركاء
        if (auth()->user()->hasRole(['admin', 'accountant'])) {
            foreach ($allPartners as $partner) {
                $adjustedPercentage = $totalShares > 0 ? ($partner->share / $totalShares) * 100 : 0;
                $partnerShares[] = [
                    'name' => $partner->name,
                    'percentage' => $partner->share,
                    'adjusted_percentage' => $adjustedPercentage,
                    'share' => $netProfit * ($adjustedPercentage / 100)
                ];
            }
        } 
        // وإلا نضيف حصة الشريك فقط
        else {
            $partner = auth()->user()->partner;
            $adjustedPercentage = $totalShares > 0 ? ($partner->share / $totalShares) * 100 : 0;
            $partnerShares[] = [
                'name' => $partner->name,
                'percentage' => $partner->share,
                'adjusted_percentage' => $adjustedPercentage,
                'share' => $netProfit * ($adjustedPercentage / 100)
            ];
        }

        $categories = [
            'income' => json_decode(Setting::get('income_categories'), true) ?? [],
            'expense' => json_decode(Setting::get('expense_categories'), true) ?? []
        ];

        return view('reports.index', compact(
            'partners',
            'categories',
            'totalIncome',
            'totalExpense',
            'netProfit',
            'partnerShares',
            'isPartner',
            'totalShares'
        ));
    }

    public function filter(Request $request)
    {
        $query = Transaction::with('partner');

        if (auth()->user()->hasRole(['admin', 'accountant'])) {
            // المدير والمحاسب يريان جميع المعاملات
            if ($request->partner_id) {
                $query->where('partner_id', $request->partner_id);
            }
        } else {
            // الشريك يرى معاملاته فقط ونسبته من المعاملات العامة
            $partnerShare = auth()->user()->partner->share ?? 0;

            $query->where(function($q) use ($partnerShare) {
                $q->where('partner_id', auth()->user()->partner_id)
                  ->orWhere(function($q2) use ($partnerShare) {
                      $q2->whereNull('partner_id')
                         ->selectRaw("*, amount * $partnerShare / 100 as share_amount");
                  });
            });
        }

        // تطبيق الفلترة حسب النوع والتاريخ
        if ($request->type) {
            $query->where('type', $request->type);
        }
        if ($request->from_date) {
            $query->whereDate('transaction_date', '>=', $request->from_date);
        }
        if ($request->to_date) {
            $query->whereDate('transaction_date', '<=', $request->to_date);
        }

        $transactions = $query->latest('transaction_date')->get();

        // حساب الإجماليات مع مراعاة نسبة الشريك للمعاملات العامة
        if (!auth()->user()->hasRole(['admin', 'accountant'])) {
            $partnerShare = auth()->user()->partner->share ?? 0;
            
            $transactions = $transactions->map(function($transaction) use ($partnerShare) {
                if (is_null($transaction->partner_id)) {
                    $transaction->amount = $transaction->amount * $partnerShare / 100;
                }
                return $transaction;
            });
        }

        // حساب الإجماليات
        $totals = [
            'income' => $transactions->where('type', 'income')->sum('amount'),
            'expense' => $transactions->where('type', 'expense')->sum('amount'),
            'net' => $transactions->where('type', 'income')->sum('amount') - 
                    $transactions->where('type', 'expense')->sum('amount')
        ];

        // حساب حصص الشركاء
        $allPartners = Partner::all();
        $totalShares = $allPartners->sum('share');
        
        $partnerShares = [];
        // دائماً نضيف جميع الشركاء للمدير والمحاسب
        if (auth()->user()->hasRole(['admin', 'accountant'])) {
            foreach ($allPartners as $partner) {
                $adjustedPercentage = $totalShares > 0 ? ($partner->share / $totalShares) * 100 : 0;
                $partnerShares[] = [
                    'name' => $partner->name,
                    'percentage' => $partner->share,
                    'adjusted_percentage' => $adjustedPercentage,
                    'share' => $totals['net'] * ($adjustedPercentage / 100)
                ];
            }
        } else {
            // للشريك العادي، نضيف حصته فقط
            $partner = auth()->user()->partner;
            if ($partner) {
                $adjustedPercentage = $totalShares > 0 ? ($partner->share / $totalShares) * 100 : 0;
                $partnerShares[] = [
                    'name' => $partner->name,
                    'percentage' => $partner->share,
                    'adjusted_percentage' => $adjustedPercentage,
                    'share' => $totals['net'] * ($adjustedPercentage / 100)
                ];
            }
        }

        $categories = [
            'income' => json_decode(Setting::get('income_categories'), true) ?? [],
            'expense' => json_decode(Setting::get('expense_categories'), true) ?? []
        ];

        return response()->json([
            'success' => true,
            'transactions' => $transactions,
            'totals' => $totals,
            'categories' => $categories,
            'partnerShares' => $partnerShares,
            'totalShares' => $totalShares,
            'isPartner' => !auth()->user()->hasRole(['admin', 'accountant'])
        ]);
    }
} 