<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use App\Models\Partner;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReportController extends Controller
{
    public function index()
    {
        $query = Transaction::query();
        
        // الشريك يرى جميع المعاملات لكن يرى حصته فقط
        $userPartner = auth()->user()->partner_id;
        $isPartner = auth()->user()->hasRole('partner');

        // لا نقيد المعاملات بالشريك لأنه يجب أن يرى الكل
        $transactions = $query->latest()->paginate(10);
        
        // حساب الإجماليات
        $totalIncome = $this->getTotalIncome($query);
        $totalExpense = $this->getTotalExpense($query);
        $netProfit = $this->getNetProfit($query);
        
        // حساب حصص الشركاء
        $partnerShares = [];
        $totalPercentage = 0;
        
        if (auth()->user()->hasRole('admin')) {
            // المدير يرى جميع الشركاء
            $partners = Partner::where('is_active', true)->get();
            $totalPercentage = $partners->sum('percentage');
            $partnerShares = $this->getAllPartnersShares($netProfit);
        } elseif ($isPartner) {
            // الشريك يرى حصته فقط
            $partner = Partner::find($userPartner);
            if ($partner) {
                $partners = Partner::where('is_active', true)->get();
                $totalPercentage = $partners->sum('percentage');
                $adjustedPercentage = $totalPercentage > 0 ? 
                    ($partner->percentage / $totalPercentage) * 100 : 0;
                    
                $partnerShares[] = [
                    'name' => $partner->name,
                    'percentage' => (float) $partner->percentage,
                    'adjusted_percentage' => (float) $adjustedPercentage,
                    'share' => (float) ($netProfit * $adjustedPercentage / 100),
                    'net_profit' => $netProfit
                ];
            }
        }
        
        return view('reports.index', [
            'transactions' => $transactions,
            'totalIncome' => $totalIncome,
            'totalExpense' => $totalExpense,
            'netProfit' => $netProfit,
            'partnerShares' => $partnerShares,
            'totalPercentage' => $totalPercentage,
            'partners' => Partner::all(), // للفلتر
            'isPartner' => $isPartner
        ]);
    }

    public function filter(Request $request)
    {
        try {
            $query = Transaction::query();

            // تطبيق الفلاتر
            if ($request->from_date) {
                $query->whereDate('transaction_date', '>=', $request->from_date);
            }
            if ($request->to_date) {
                $query->whereDate('transaction_date', '<=', $request->to_date);
            }
            if ($request->type) {
                $query->where('type', $request->type);
            }

            // الحصول على المعاملات
            $transactions = $query->get();

            // حساب الإحصائيات بدقة
            $stats = [
                'total_income' => floor($transactions->where('type', 'income')->sum('amount') * 1000) / 1000,
                'total_expense' => floor($transactions->where('type', 'expense')->sum('amount') * 1000) / 1000,
                'net_profit' => 0
            ];
            
            $stats['net_profit'] = floor(($stats['total_income'] - $stats['total_expense']) * 1000) / 1000;

            // حساب حصص الشركاء
            $partners = Partner::where('is_active', true)->get();
            $totalPercentage = $partners->sum('percentage');
            $remainingAmount = $stats['net_profit'];
            $partnerShares = collect();

            if (auth()->user()->hasRole('admin')) {
                // المدير يرى جميع الشركاء
                foreach ($partners as $index => $partner) {
                    if ($index === $partners->count() - 1) {
                        $share = $remainingAmount;
                    } else {
                        $share = floor(($stats['net_profit'] * $partner->percentage) / 100 * 1000) / 1000;
                        $remainingAmount -= $share;
                    }

                    $partnerShares->push([
                        'name' => $partner->name,
                        'percentage' => (float) $partner->percentage,
                        'adjusted_percentage' => (float) $partner->percentage,
                        'share' => $share
                    ]);
                }
            } elseif (auth()->user()->hasRole('partner')) {
                // الشريك يرى حصته فقط
                $partner = Partner::find(auth()->user()->partner_id);
                if ($partner) {
                    $share = floor(($stats['net_profit'] * $partner->percentage) / 100 * 1000) / 1000;
                    
                    $partnerShares->push([
                        'name' => $partner->name,
                        'percentage' => (float) $partner->percentage,
                        'adjusted_percentage' => (float) $partner->percentage,
                        'share' => $share
                    ]);

                    // تعديل النسبة الكلية لتكون نسبة الشريك فقط
                    $totalPercentage = $partner->percentage;
                }
            }

            return response()->json([
                'success' => true,
                'transactions' => $transactions->map(function($transaction) {
                    return [
                        'transaction_date' => $transaction->transaction_date,
                        'type' => $transaction->type,
                        'category' => $transaction->category,
                        'amount' => floor($transaction->amount * 1000) / 1000,
                        'description' => $transaction->description,
                        'partner' => $transaction->partner ? [
                            'name' => $transaction->partner->name
                        ] : null
                    ];
                }),
                'stats' => $stats,
                'partnerShares' => $partnerShares,
                'totalPercentage' => $totalPercentage,
                'isPartner' => auth()->user()->hasRole('partner')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء تصفية البيانات'
            ], 422);
        }
    }

    private function getTotalIncome($query = null)
    {
        $query = $query ?? Transaction::query();
        return floor($query->clone()->where('type', 'income')->sum('amount') * 1000) / 1000;
    }

    private function getTotalExpense($query = null)
    {
        $query = $query ?? Transaction::query();
        return floor($query->clone()->where('type', 'expense')->sum('amount') * 1000) / 1000;
    }

    private function getNetProfit($query = null)
    {
        return floor(($this->getTotalIncome($query) - $this->getTotalExpense($query)) * 1000) / 1000;
    }

    public function export()
    {
        // تنفيذ التصدير
    }

    private function getAllPartnersShares($netProfit)
    {
        $partners = Partner::where('is_active', true)->get();
        $totalPercentage = $partners->sum('percentage');
        $remainingAmount = $netProfit;
        $result = [];

        foreach ($partners as $index => $partner) {
            if ($index === $partners->count() - 1) {
                // آخر شريك يأخذ المبلغ المتبقي لتجنب فروقات التقريب
                $share = $remainingAmount;
            } else {
                $share = floor(($netProfit * $partner->percentage) / 100 * 1000) / 1000;
                $remainingAmount -= $share;
            }

            $result[] = [
                'name' => $partner->name,
                'percentage' => (float) $partner->percentage,
                'adjusted_percentage' => (float) $partner->percentage,
                'share' => $share
            ];
        }

        return $result;
    }
} 