<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class ProfileController extends Controller
{
    public function edit()
    {
        return view('profile.edit', [
            'user' => auth()->user()
        ]);
    }

    public function update(Request $request)
    {
        $user = auth()->user();
        
        try {
            $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|string|email|max:255|unique:users,email,'.$user->id,
                'current_password' => 'nullable|required_with:password',
                'password' => 'nullable|min:6|confirmed',
                'avatar' => 'nullable|image|mimes:jpeg,png,jpg|max:2048'
            ]);

            // تحديث البيانات الأساسية
            $user->name = $request->name;
            $user->email = $request->email;

            // تحديث كلمة المرور إذا تم تقديمها
            if ($request->filled('current_password')) {
                if (!Hash::check($request->current_password, $user->password)) {
                    return back()->withErrors(['current_password' => 'كلمة المرور الحالية غير صحيحة']);
                }
                
                if ($request->filled('password')) {
                    $user->password = Hash::make($request->password);
                }
            }

            // معالجة الصورة
            if ($request->hasFile('avatar')) {
                try {
                    // التأكد من وجود المجلد
                    if (!Storage::disk('public')->exists('avatars')) {
                        Storage::disk('public')->makeDirectory('avatars');
                    }

                    // حذف الصورة القديمة إذا وجدت
                    if ($user->avatar && Storage::disk('public')->exists($user->avatar)) {
                        Storage::disk('public')->delete($user->avatar);
                    }

                    // حفظ الصورة الجديدة
                    $file = $request->file('avatar');
                    $filename = 'avatar_' . $user->id . '_' . time() . '.' . $file->getClientOriginalExtension();
                    $path = $file->storeAs('avatars', $filename, 'public');
                    $user->avatar = $path;
                } catch (\Exception $e) {
                    Log::error('خطأ في تحميل الصورة: ' . $e->getMessage());
                    return back()->with('error', 'حدث خطأ أثناء تحميل الصورة');
                }
            }

            $user->save();

            return back()->with('success', 'تم تحديث الملف الشخصي بنجاح');
        } catch (\Exception $e) {
            Log::error('خطأ في تحديث الملف الشخصي: ' . $e->getMessage());
            return back()->with('error', 'حدث خطأ أثناء تحديث الملف الشخصي: ' . $e->getMessage());
        }
    }
} 